#  Copyright 2022 United Kingdom Research and Innovation
#  Copyright 2022 The University of Manchester
#
#  Licensed under the Apache License, Version 2.0 (the "License");
#  you may not use this file except in compliance with the License.
#  You may obtain a copy of the License at
#
#      http://www.apache.org/licenses/LICENSE-2.0
#
#  Unless required by applicable law or agreed to in writing, software
#  distributed under the License is distributed on an "AS IS" BASIS,
#  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
#  See the License for the specific language governing permissions and
#  limitations under the License.
#
# Authors:
# CIL Developers, listed at: https://github.com/TomographicImaging/CIL/blob/master/NOTICE.txt

import numpy as np
from cil.optimisation.operators import LinearOperator
from cil.utilities import dataexample
from cil.framework import AcquisitionGeometry, ImageGeometry
from cil.framework.labels import AcquisitionDimension, AcquisitionType

def create_cone_flex_default_ig(ag):
    '''
    Create a default ImageGeometry for Cone3D_Flex geometries.
    This makes use of the mean magnification value to scale the pixel size when determining the voxel size.

    Parameters
    ----------
    ag: AcquisitionGeometry
        Must be a Cone3D_Flex AcquisitionGeometry
    
    Returns
    -------
    ImageGeometry
    '''
    mag = ag.config.system.calculate_magnification()[2]
    mean_mag = np.mean(mag)
    num_voxel_xy = int(np.ceil(ag.config.panel.num_pixels[0]))
    voxel_size_xy = ag.config.panel.pixel_size[0] / mean_mag

    num_voxel_z = int(np.ceil(ag.config.panel.num_pixels[1]))
    voxel_size_z = ag.config.panel.pixel_size[1] / mean_mag
    ig = ImageGeometry(num_voxel_xy, num_voxel_xy, num_voxel_z, voxel_size_xy, voxel_size_xy, voxel_size_z)
    return ig



class SimData(object):

    def _get_roi_3D(self):
        roi = [20,30,40]
        center_offset = [50,-20,5]

        self.ig_roi = self.ig.copy()
        self.ig_roi.voxel_num_x = roi[2]
        self.ig_roi.voxel_num_y = roi[1]
        self.ig_roi.voxel_num_z = roi[0]
        self.ig_roi.center_x = center_offset[2]*self.ig_roi.voxel_size_x
        self.ig_roi.center_y = center_offset[1]*self.ig_roi.voxel_size_y
        self.ig_roi.center_z = center_offset[0]*self.ig_roi.voxel_size_z

        index_roi = [None]*3
        for i in range(3):
            ind0 = center_offset[i] + (self.img_data.shape[i] - roi[i])//2
            ind1 = ind0 + roi[i]
            index_roi[i] = (ind0, ind1)

        self.gold_roi = self.img_data.array[index_roi[0][0]:index_roi[0][1],index_roi[1][0]:index_roi[1][1],index_roi[2][0]:index_roi[2][1]]

        # single slice roi
        self.ig_single_slice = self.ig.copy()
        slice_index = 20
        slice_offset = -(self.ig.voxel_num_z-1)/2 + slice_index
        self.ig_single_slice.voxel_num_z = 1
        self.ig_single_slice.center_z = slice_offset*self.ig_roi.voxel_size_z

        self.gold_roi_single_slice = self.img_data.array[slice_index,:,:]


    def _get_roi_2D(self):
        roi = [30,40]
        center_offset = [-20,5]

        self.ig_roi = self.ig.copy()
        self.ig_roi.voxel_num_x = roi[1]
        self.ig_roi.voxel_num_y = roi[0]
        self.ig_roi.center_x = center_offset[1]*self.ig_roi.voxel_size_x
        self.ig_roi.center_y = center_offset[0]*self.ig_roi.voxel_size_y

        index_roi = [None]*2
        for i in range(2):
            ind0 = center_offset[i] + (self.img_data.shape[i] - roi[i])//2
            ind1 = ind0 + roi[i]
            index_roi[i] = (ind0, ind1)

        self.gold_roi = self.img_data.array[index_roi[0][0]:index_roi[0][1],index_roi[1][0]:index_roi[1][1]]


    def Cone3D(self):
        self.acq_data = dataexample.SIMULATED_CONE_BEAM_DATA.get()
        self.acq_data.reorder(self.backend)

        self.img_data = dataexample.SIMULATED_SPHERE_VOLUME.get()

        self.acq_data=np.log(self.acq_data)
        self.acq_data*=-1.0

        self.ig = self.img_data.geometry
        self.ag = self.acq_data.geometry

        self._get_roi_3D()


    def Cone3DFlex(self):

        self.acq_data = dataexample.SIMULATED_CONE_BEAM_DATA.get()
        self.acq_data.reorder(self.backend)

        self.img_data = dataexample.SIMULATED_SPHERE_VOLUME.get()

        self.acq_data=np.log(self.acq_data)
        self.acq_data*=-1.0

        self.ig = self.img_data.geometry

        # convert geometry to Flex
        system = self.acq_data.geometry.config.system
        src = system.source.position
        det_pos = system.detector.position
        det_dir_x = system.detector.direction_x
        det_dir_y = system.detector.direction_y
        obj_pos = system.rotation_axis.position
        obj_dir = system.rotation_axis.direction
        angles = self.acq_data.geometry.config.angles.angle_data

        src_pos_set = [None]*len(angles)
        det_pos_set = [None]*len(angles)
        det_dir_x_set = [None]*len(angles)
        det_dir_y_set = [None]*len(angles)

        for i, ang in enumerate(angles):
            ang_rad = -np.deg2rad(ang)
            # rotation matrix
            RotationMatrix = np.eye(3)
            RotationMatrix[0,0] = RotationMatrix[1,1] = np.cos(ang_rad)
            RotationMatrix[0,1] = -np.sin(ang_rad)
            RotationMatrix[1,0] = np.sin(ang_rad)
            
            src_pos_set[i] = RotationMatrix.dot(src - obj_pos) + obj_pos
            det_pos_set[i] = RotationMatrix.dot(det_pos - obj_pos) + obj_pos
            det_dir_x_set[i] = RotationMatrix.dot(det_dir_x)
            det_dir_y_set[i] = RotationMatrix.dot(det_dir_y)

        ag_new = AcquisitionGeometry.create_Cone3D_Flex(src_pos_set, det_pos_set, det_dir_x_set, det_dir_y_set)
        ag_new.set_panel([self.acq_data.geometry.pixel_num_h, self.acq_data.geometry.pixel_num_v], [self.acq_data.geometry.pixel_size_h, self.acq_data.geometry.pixel_size_v], origin='bottom-left')
        # Flex geometry has the same labels except uses PROJECTION instead of ANGLE:
        new_labels = list(self.acq_data.dimension_labels)
        new_labels[1] = AcquisitionDimension.PROJECTION
        ag_new.set_labels(new_labels)

        self.ag = ag_new
        self._get_roi_3D()


    def Parallel3D(self):
        self.acq_data = dataexample.SIMULATED_PARALLEL_BEAM_DATA.get()
        self.acq_data.reorder(self.backend)

        self.img_data = dataexample.SIMULATED_SPHERE_VOLUME.get()

        self.acq_data=np.log(self.acq_data)
        self.acq_data*=-1.0

        self.ig = self.img_data.geometry
        self.ag = self.acq_data.geometry

        self._get_roi_3D()


    def Cone2D(self):

        self.acq_data = dataexample.SIMULATED_CONE_BEAM_DATA.get().get_slice(vertical='centre')
        self.acq_data.reorder(self.backend)

        self.img_data = dataexample.SIMULATED_SPHERE_VOLUME.get().get_slice(vertical='centre')

        self.acq_data=np.log(self.acq_data)
        self.acq_data*=-1.0

        self.ig = self.img_data.geometry
        self.ag = self.acq_data.geometry

        self._get_roi_2D()


    def Parallel2D(self):

        self.acq_data = dataexample.SIMULATED_PARALLEL_BEAM_DATA.get().get_slice(vertical='centre')
        self.acq_data.reorder(self.backend)

        self.img_data = dataexample.SIMULATED_SPHERE_VOLUME.get().get_slice(vertical='centre')

        self.acq_data=np.log(self.acq_data)
        self.acq_data*=-1.0

        self.ig = self.img_data.geometry
        self.ag = self.acq_data.geometry

        self._get_roi_2D()


class TestCommon_ProjectionOperator_TOY(object):
    '''
    Tests behaviour of the operators on a variety of geometries.
    '''
    def Cone3D(self):
        '''
            These are all single cone beam projection geometries. Pixels of  1, 2, 0.5, 0.5, Voxels of 1, 2, 0.5, 0.25
        '''

        self.test_geometries=[]
        ag_test_1 = AcquisitionGeometry.create_Cone3D(source_position=[0,-1000,0],detector_position=[0,0,0])\
                                            .set_panel([16,16],[1,1])\
                                            .set_angles([0])
        ag_test_1.set_labels(AcquisitionDimension.get_order_for_engine(self.backend, ag_test_1))


        ig_test_1 = ag_test_1.get_ImageGeometry()
        norm_1 = 4
        self.test_geometries.append((ag_test_1, ig_test_1, 4))


        ag_test_2 = AcquisitionGeometry.create_Cone3D(source_position=[0,-1000,0],detector_position=[0,0,0])\
                                            .set_panel([16,16],[2,2])\
                                            .set_angles([0])
        ag_test_2.set_labels(AcquisitionDimension.get_order_for_engine(self.backend, ag_test_2))

        ig_test_2 = ag_test_2.get_ImageGeometry()
        norm_2 = 8
        self.test_geometries.append((ag_test_2, ig_test_2, norm_2))


        ag_test_3 = AcquisitionGeometry.create_Cone3D(source_position=[0,-1000,0],detector_position=[0,0,0])\
                                            .set_panel([16,16],[0.5,0.5])\
                                            .set_angles([0])
        ag_test_3.set_labels(AcquisitionDimension.get_order_for_engine(self.backend, ag_test_3))
        ig_test_3 = ag_test_3.get_ImageGeometry()

        norm_3 = 2
        self.test_geometries.append((ag_test_3, ig_test_3, norm_3))


        ag_test_4 = AcquisitionGeometry.create_Cone3D(source_position=[0,-1000,0],detector_position=[0,1000,0])\
                                            .set_panel([16,16],[0.5,0.5])\
                                            .set_angles([0])
        ag_test_4.set_labels(AcquisitionDimension.get_order_for_engine(self.backend, ag_test_4))
        ig_test_4 = ag_test_4.get_ImageGeometry()

        norm_4 = 1
        self.test_geometries.append((ag_test_4, ig_test_4, norm_4))


    def Cone2D(self):
        '''
            These are all single cone beam projection geometries. Pixels of  1, 2, 0.5, 0.5, Voxels of 1, 2, 0.5, 0.25
        '''

        self.test_geometries=[]
        ag_test_1 = AcquisitionGeometry.create_Cone2D(source_position=[0,-1000],detector_position=[0,0])\
                                            .set_panel(16,1)\
                                            .set_angles([0])
        ag_test_1.set_labels(AcquisitionDimension.get_order_for_engine(self.backend, ag_test_1))


        ig_test_1 = ag_test_1.get_ImageGeometry()
        norm_1 = 4
        self.test_geometries.append((ag_test_1, ig_test_1, 4))


        ag_test_2 = AcquisitionGeometry.create_Cone2D(source_position=[0,-1000],detector_position=[0,0])\
                                            .set_panel(16,2)\
                                            .set_angles([0])
        ag_test_2.set_labels(AcquisitionDimension.get_order_for_engine(self.backend, ag_test_2))

        ig_test_2 = ag_test_2.get_ImageGeometry()
        norm_2 = 8
        self.test_geometries.append((ag_test_2, ig_test_2, norm_2))


        ag_test_3 = AcquisitionGeometry.create_Cone2D(source_position=[0,-1000],detector_position=[0,0])\
                                            .set_panel(16,0.5)\
                                            .set_angles([0])
        ag_test_3.set_labels(AcquisitionDimension.get_order_for_engine(self.backend, ag_test_3))
        ig_test_3 = ag_test_3.get_ImageGeometry()

        norm_3 = 2
        self.test_geometries.append((ag_test_3, ig_test_3, norm_3))


        ag_test_4 = AcquisitionGeometry.create_Cone2D(source_position=[0,-1000],detector_position=[0,1000])\
                                            .set_panel(16,0.5)\
                                            .set_angles([0])
        ag_test_4.set_labels(AcquisitionDimension.get_order_for_engine(self.backend, ag_test_4))
        ig_test_4 = ag_test_4.get_ImageGeometry()

        norm_4 = 1
        self.test_geometries.append((ag_test_4, ig_test_4, norm_4))


    def Parallel3D(self):
        '''
            These are all single parallel beam projection geometries. Pixels & voxels of  1, 2, 0.5
        '''

        self.test_geometries=[]
        ag_test_1 = AcquisitionGeometry.create_Parallel3D()\
                                            .set_panel([16,16],[1,1])\
                                            .set_angles([0])
        ag_test_1.set_labels(AcquisitionDimension.get_order_for_engine(self.backend, ag_test_1))


        ig_test_1 = ag_test_1.get_ImageGeometry()
        norm_1 = 4
        self.test_geometries.append((ag_test_1, ig_test_1, norm_1))

        ag_test_2 = AcquisitionGeometry.create_Parallel3D()\
                                            .set_panel([16,16],[2,2])\
                                            .set_angles([0])
        ag_test_2.set_labels(AcquisitionDimension.get_order_for_engine(self.backend, ag_test_2))


        ig_test_2 = ag_test_2.get_ImageGeometry()
        norm_2 = 8
        self.test_geometries.append((ag_test_2, ig_test_2, norm_2))


        ag_test_3 = AcquisitionGeometry.create_Parallel3D()\
                                            .set_panel([16,16],[0.5,0.5])\
                                            .set_angles([0])
        ag_test_3.set_labels(AcquisitionDimension.get_order_for_engine(self.backend, ag_test_3))


        ig_test_3 = ag_test_3.get_ImageGeometry()
        norm_3 = 2
        self.test_geometries.append((ag_test_3, ig_test_3, norm_3))


    def Parallel2D(self):
        '''
            These are all single parallel beam projection geometries. Pixels & voxels of  1, 2, 0.5
        '''

        self.test_geometries=[]
        ag_test_1 = AcquisitionGeometry.create_Parallel2D()\
                                            .set_panel(16,1)\
                                            .set_angles([0])

        ag_test_1.set_labels(AcquisitionDimension.get_order_for_engine(self.backend, ag_test_1))

        ig_test_1 = ag_test_1.get_ImageGeometry()
        norm_1 = 4
        self.test_geometries.append((ag_test_1, ig_test_1, norm_1))

        ag_test_2 = AcquisitionGeometry.create_Parallel2D()\
                                            .set_panel(16,2)\
                                            .set_angles([0])
        ag_test_2.set_labels(AcquisitionDimension.get_order_for_engine(self.backend, ag_test_2))


        ig_test_2 = ag_test_2.get_ImageGeometry()
        norm_2 = 8
        self.test_geometries.append((ag_test_2, ig_test_2, norm_2))


        ag_test_3 = AcquisitionGeometry.create_Parallel2D()\
                                            .set_panel(16,0.5)\
                                            .set_angles([0])
        ag_test_3.set_labels(AcquisitionDimension.get_order_for_engine(self.backend, ag_test_3))

        ig_test_3 = ag_test_3.get_ImageGeometry()
        norm_3 = 2
        self.test_geometries.append((ag_test_3, ig_test_3, norm_3))

    def Cone3DFlex(self):
        '''
            These are all single cone beam projection geometries. Pixels of  1, 2, 0.5, 0.5, Voxels of 1, 2, 0.5, 0.25
        '''

        self.test_geometries=[]
        ag_test_1 = AcquisitionGeometry.create_Cone3D_Flex(source_position_set=[[0,-1000,0]],detector_position_set=[[0,0,0]], detector_direction_x_set=[[1, 0, 0]],detector_direction_y_set=[[0, 0, 1]], volume_centre_position=[0,0,0])\
                                            .set_panel([16,16],[1,1])
        ag_test_1.set_labels(AcquisitionDimension.get_order_for_engine(self.backend, ag_test_1))

        ig_test_1 = create_cone_flex_default_ig(ag_test_1)
        norm_1 = 4
        self.test_geometries.append((ag_test_1, ig_test_1, 4))


        ag_test_2 = AcquisitionGeometry.create_Cone3D_Flex(source_position_set=[[0,-1000,0]],detector_position_set=[[0,0,0]], detector_direction_x_set=[[1, 0, 0]],detector_direction_y_set=[[0, 0, 1]], volume_centre_position=[0,0,0])\
                                            .set_panel([16,16],[2,2])
        ag_test_2.set_labels(AcquisitionDimension.get_order_for_engine(self.backend, ag_test_2))

        ig_test_2 = create_cone_flex_default_ig(ag_test_2)
        norm_2 = 8
        self.test_geometries.append((ag_test_2, ig_test_2, norm_2))

        ag_test_3 = AcquisitionGeometry.create_Cone3D_Flex(source_position_set=[[0,-1000,0]],detector_position_set=[[0,0,0]], detector_direction_x_set=[[1, 0, 0]],detector_direction_y_set=[[0, 0, 1]], volume_centre_position=[0,0,0])\
                                            .set_panel([16,16],[0.5,0.5])
        ag_test_3.set_labels(AcquisitionDimension.get_order_for_engine(self.backend, ag_test_3))
        ig_test_3 = create_cone_flex_default_ig(ag_test_3)

        norm_3 = 2
        self.test_geometries.append((ag_test_3, ig_test_3, norm_3))

        ag_test_4 = AcquisitionGeometry.create_Cone3D_Flex(source_position_set=[[0,-1000,0]],detector_position_set=[[0,1000,0]], detector_direction_x_set=[[1, 0, 0]],detector_direction_y_set=[[0, 0, 1]], volume_centre_position=[0,0,0])\
                                            .set_panel([16,16],[0.5,0.5])
        ag_test_4.set_labels(AcquisitionDimension.get_order_for_engine(self.backend, ag_test_4))
        ig_test_4 = create_cone_flex_default_ig(ag_test_4)

        norm_4 = 1
        self.test_geometries.append((ag_test_4, ig_test_4, norm_4))

    def test_norm(self):
        count = 1
        for ag, ig, norm in self.test_geometries:
            Op = self.ProjectionOperator(ig, ag,  **self.PO_args)
            n = Op.norm()

            diff =abs(n-norm)
            self.assertLess(diff, self.tolerance_norm, "Geometry: {0} Calculated norm {1}, expected {2}".format(count, n, norm))


    def test_linearity(self):
        count = 1
        for ag, ig, _ in self.test_geometries:
            Op = self.ProjectionOperator(ig, ag,  **self.PO_args)
            res = LinearOperator.dot_test(Op, tolerance=self.tolerance_linearity)
            self.assertTrue(res, "Geometry: {0} Dot-test out of tolerance {1}".format(count, self.tolerance_linearity))
            count +=1


class TestCommon_ProjectionOperator(object):

    '''
    These are all single projection geometries approximating parallel beam.
    '''

    def Cone3D(self):
        self.ag = AcquisitionGeometry.create_Cone3D(source_position=[0,-100000,0],detector_position=[0,0,0],)\
                                            .set_panel([16,16],[1,1])\
                                            .set_angles([0])\
                                            .set_labels(['vertical','horizontal'])

    def Cone2D(self):
        self.ag = AcquisitionGeometry.create_Cone2D(source_position=[0,-100000],detector_position=[0,0])\
                                            .set_panel(16,1)\
                                            .set_angles([0])\
                                            .set_labels(['horizontal'])

    def Parallel3D(self):
        self.ag = AcquisitionGeometry.create_Parallel3D()\
                                            .set_panel([16,16],[1,1])\
                                            .set_angles([0])\
                                            .set_labels(['vertical','horizontal'])

    def Parallel2D(self):
        self.ag = AcquisitionGeometry.create_Parallel2D()\
                                            .set_panel(16,1)\
                                            .set_angles([0])\
                                            .set_labels(['horizontal'])

    def Cone3DFlex(self):
        self.ag = AcquisitionGeometry.create_Cone3D_Flex(source_position_set=[[0,-100000,0]],detector_position_set=[[0,0,0]], detector_direction_x_set=[[1, 0, 0]],detector_direction_y_set=[[0, 0, 1]], volume_centre_position=[0,0,0])\
                                            .set_panel([16,16],[1,1])\
                                            .set_labels(['vertical','horizontal'])
        
    def test_forward_projector(self):

        #create checker-board image
        res = np.zeros((16,16,16))
        ones = np.ones((4,16,4))
        for k in range(4):
            for i in range(4):
                if (i + k)% 2 == 0:
                    res[k*4:(k+1)*4,:,i*4:(i+1)*4] = ones

        #create checker-board forward projection for parallel rays
        checker = np.zeros((16,16))
        ones = np.ones((4,4))
        for j in range(4):
            for i in range(4):
                if (i + j)% 2 == 0:
                    checker[j*4:(j+1)*4,i*4:(i+1)*4] = ones * 16

        if AcquisitionType.DIM2 & self.ag.dimension:
            checker = checker[0]
            res = res[0]

        if self.ag.geom_type & AcquisitionType.CONE_FLEX:
            ig = create_cone_flex_default_ig(self.ag)
        else:
            ig = self.ag.get_ImageGeometry()
        
        volume = ig.allocate(0)

        volume.fill(res)

        Op = self.ProjectionOperator(ig, self.ag, **self.PO_args)
        fp = Op.direct(volume)

        np.testing.assert_allclose(fp.array, checker, atol = self.tolerance_fp)

    def test_backward_projector(self):

        #create checker-board projection
        checker = np.zeros((16,16))
        ones = np.ones((4,4))
        for j in range(4):
            for i in range(4):
                if (i + j)% 2 == 0:
                    checker[j*4:(j+1)*4,i*4:(i+1)*4] = ones

        #create backprojection of checker-board
        res = np.zeros((16,16,16))
        ones = np.ones((4,16,4))
        for k in range(4):
            for i in range(4):
                if (i + k)% 2 == 0:
                    res[k*4:(k+1)*4,:,i*4:(i+1)*4] = ones

        if AcquisitionType.DIM2 & self.ag.dimension:
            checker = checker[0]
            res = res[0]

        if self.ag.geom_type & AcquisitionType.CONE_FLEX:
            ig = create_cone_flex_default_ig(self.ag)
        else:
            ig = self.ag.get_ImageGeometry()
        
        data = self.ag.allocate(0)

        data.fill(checker)

        Op = self.ProjectionOperator(ig, self.ag, **self.PO_args)
        bp = Op.adjoint(data)

        if self.ag.geom_type == 'cone' or self.ag.geom_type == 'cone_flex':
            #as cone beam res is not perfect grid
            np.testing.assert_allclose(bp.array, res, atol=1e-3)
        else:
            np.testing.assert_equal(bp.array, res)


class TestCommon_ProjectionOperator_SIM(SimData):
    '''
    Tests forward and backward operators function with and without 'out'
    '''

    def test_forward_projector(self):
        Op = self.ProjectionOperator(self.ig, self.ag, **self.PO_args)
        fp = Op.direct(self.img_data)
        np.testing.assert_allclose(fp.as_array(), self.acq_data.as_array(),atol=self.tolerance_fp)

        fp2 = fp.copy()
        fp2.fill(0)
        fp3 = Op.direct(self.img_data,out=fp2)
        np.testing.assert_allclose(fp.as_array(), fp2.as_array(),1e-8)
        np.testing.assert_equal(id(fp2), id(fp3))


    def test_forward_projector_flipped(self):
        ag = self.ag.copy()
        ag.set_panel([self.ag.pixel_num_h,self.ag.pixel_num_v],[self.ag.pixel_size_h,self.ag.pixel_size_v],origin='top-right')
        Op = self.ProjectionOperator(self.ig, ag, **self.PO_args)
        fp = Op.direct(self.img_data)

        axes = [i for i, label in enumerate(self.acq_data.dimension_labels) if label != 'angle']
        gold = self.acq_data.copy()
        gold.fill(np.flip(self.acq_data.as_array(), axis=axes))

       # show2D([self.acq_data, fp, gold],title=['self.acq_data', 'fp', 'gold']).save('test0.png')
        np.testing.assert_allclose(fp.as_array(), gold.as_array(),atol=self.tolerance_fp)

        fp2 = fp.copy()
        fp2.fill(0)
        Op.direct(self.img_data,out=fp2)

        np.testing.assert_allclose(fp.as_array(), fp2.as_array(),1e-8)


    def test_backward_projectors_functionality(self):
        #this checks mechanics but not value
        Op = self.ProjectionOperator(self.ig, self.ag, **self.PO_args)
        bp = Op.adjoint(self.acq_data)

        bp2 = bp.copy()
        bp2.fill(0)
        bp3 = Op.adjoint(self.acq_data,out=bp2)
        np.testing.assert_allclose(bp.as_array(), bp2.as_array(), 1e-8)
        np.testing.assert_equal(id(bp2), id(bp3))


    def test_input_arguments(self):

        #default image_geometry, named parameter acquisition_geometry
        if not (self.ag.geom_type & AcquisitionType.CONE_FLEX):
            Op = self.ProjectionOperator(acquisition_geometry=self.ag, **self.PO_args)
            fp = Op.direct(self.img_data)
            np.testing.assert_allclose(fp.as_array(), self.acq_data.as_array(),atol=self.tolerance_fp)


class TestCommon_FBP_SIM(SimData):
    '''
    FBP tests on simulated data
    '''
    def test_FBP(self):

        FBP = self.FBP(self.ig, self.ag, **self.FBP_args)
        reco = FBP(self.acq_data)
        np.testing.assert_allclose(reco.as_array(), self.img_data.as_array(),atol=self.tolerance_fbp)

        reco2 = reco.copy()
        reco2.fill(0)
        FBP(self.acq_data,out=reco2)
        np.testing.assert_allclose(reco.as_array(), reco2.as_array(),atol=1e-8)

    def test_FBP_flipped(self):

        data = self.acq_data.copy()
        data.geometry.set_panel([self.ag.pixel_num_h,self.ag.pixel_num_v],[self.ag.pixel_size_h,self.ag.pixel_size_v],origin='top-right')

        axes = [i for i, label in enumerate(self.acq_data.dimension_labels) if label != 'angle']
        data.fill(np.flip(self.acq_data.as_array(), axis=axes))

        FBP = self.FBP(self.ig, data.geometry, **self.FBP_args)
        reco = FBP(data)

        #show2D([self.img_data, reco],title=['self.img_data', 'reco']).save('test4.png')

        np.testing.assert_allclose(reco.as_array(), self.img_data.as_array(), atol=self.tolerance_fbp)

        reco2 = reco.copy()
        reco2.fill(0)
        FBP(data,out=reco2)
        np.testing.assert_allclose(reco.as_array(), reco2.as_array(),atol=1e-8)


    def test_FBP_roi(self):
        FBP = self.FBP(self.ig_roi, self.ag, **self.FBP_args)
        reco = FBP(self.acq_data)

        #show2D([reco, self.gold_roi]).save('test0_Flex.png')
        np.testing.assert_allclose(reco.as_array(), self.gold_roi, atol=self.tolerance_fbp_roi)

        if AcquisitionType.DIM3 & self.ag.dimension:
            FBP = self.FBP(self.ig_single_slice, self.ag, **self.FBP_args)
            reco = FBP(self.acq_data)
            np.testing.assert_allclose(reco.as_array(), self.gold_roi_single_slice, atol=self.tolerance_fbp_roi)


    def test_input_arguments(self):
        # default image_geometry, named parameter acquisition_geometry
        # No default ig for cone_flex so we skip this test in this case
        if not (self.ag.geom_type & AcquisitionType.CONE_FLEX): 
            FBP = self.FBP(acquisition_geometry = self.ag, **self.FBP_args)
            reco = FBP(self.acq_data)
            np.testing.assert_allclose(reco.as_array(), self.img_data.as_array(),atol=self.tolerance_fbp)


class TestCommon_ProjectionOperatorBlockOperator(object):
    # def setUp(self):
    #     data = dataexample.SIMULATED_PARALLEL_BEAM_DATA.get()
    #     self.data = data.get_slice(vertical='centre')
    #     K = ProjectionOperator(image_geometry=ig, acquisition_geometry=data.geometry)
    #     A = ProjectionOperator(image_geometry=ig, acquisition_geometry=self.data.geometry)
    #     self.projectionOperator = (A, K)
    def partition_test(self):

        A, K = self.projectionOperator

        u = A.adjoint(self.data)
        v = K.adjoint(self.datasplit)

        # the images are not entirely the same as the BlockOperator's requires to
        # add all the data of the adjoint operator, which may result in a slightly
        # different image
        np.testing.assert_allclose(u.as_array(), v.as_array(), rtol=1.2e-6, atol=1.6e-4)

        # test if using data output

        v.fill(0)
        v2 = K.adjoint(self.datasplit, out=v)
        np.testing.assert_allclose(u.as_array(), v.as_array(), rtol=1.2e-6, atol=1.6e-4)
        np.testing.assert_equal(id(v), id(v2))



        x = A.direct(u)
        y = K.direct(v)

        # let's check that the data is the same
        def check_data_is_the_same(x,y):
            k = 0
            wrong = 0
            for el in y.containers:
                for j in range(el.shape[0]):
                    try:
                        np.testing.assert_allclose(el.as_array()[j], x.as_array()[k], atol=7e-2, rtol=1e-6)
                    except AssertionError as ae:
                        print(ae)
                        wrong += 1
                    # show2D([el.as_array()[j], x.as_array()[k]], cmap=['inferno', 'inferno'])
                    k += 1

            assert wrong == 0

            # reassemlbe the data
            out = x * 0
            k = 0
            for i, el in enumerate(y.containers):
                # print (i, el.shape)
                for j in range(el.shape[0]):
                    out.array[k] = el.as_array()[j]
                    k += 1

            # show2D([out, x, out-x], cmap=['inferno', 'inferno', 'seismic'], title=['out', 'x', 'diff'], \
            #     num_cols=3)
            np.testing.assert_allclose(out.as_array(), x.as_array(), atol=1e-2, rtol=1e-6)

        check_data_is_the_same(x,y)
        y.fill(0)
        y2 = K.direct(v, out=y)
        check_data_is_the_same(x,y)
        np.testing.assert_equal(id(y), id(y2))
